﻿#using System.Threading.Tasks;

#using Tessa.Cards;
#using Tessa.Extensions.Default.Server.Workflow.KrProcess;
#using Tessa.Extensions.Default.Shared.Workflow.KrProcess;
#using Tessa.Extensions.Default.Shared.Workflow.KrProcess.ClientCommandInterpreter;

/// <summary>
/// Асинхронно сохраняет и открывает на клиенте инициализируемую карточку служебной записки.
/// </summary>
/// <param name="dialog">Контекст скриптов обработки диалогов в Workflow Engine.</param>
/// <returns>Асинхронная задача.</returns>
protected async Task StoreAndOpenCardAsync(WorkflowDialogContext dialog)
{
	var fileManager = this.Context.Container.Resolve<ICardFileManager>();
	var tokenProvider = this.Context.Container.Resolve<IKrTokenProvider>();

	var dialogCard = await dialog.GetCardObjectAsync();

	// Обработка файлов приложенных к инициализируемой карточке служебной записки.
	await using (var container = await fileManager.CreateContainerAsync(dialogCard, cancellationToken: this.Context.CancellationToken))
	{
		// Подготовка контента приложенных и/или изменённых файлов для последующего сохранения.
		foreach(var dFile in container.FileContainer.Files)
		{
			if(!dFile.Content.HasData)
			{
				await dFile.ReplaceAsync(await dialog.GetFileContentAsync(dialogCard.Files.Single(i => i.RowID == dFile.ID)));
			}
		}
		
		var storeResponse = await container.StoreAsync((c, request, ct) =>
		{
			tokenProvider.CreateToken(request.Card.ID).Set(request.Card.Info);
			return new ValueTask();
		},
		cancellationToken: this.Context.CancellationToken);
		
		this.ValidationResult.Add(storeResponse.ValidationResult);
		if (!storeResponse.ValidationResult.IsSuccessful())
		{
			return;
		}
	}

	// Добавление клиентской команды на открытие инициализированной карточки во вкладке Tessa Client.
	if (!(this.Context.ResponseInfo.TryGetValue(KrProcessSharedExtensions.KrProcessClientCommandInfoMark, out var commandsObj)
		&& commandsObj is List<object> commands))
	{
		commands = new List<object>();
		this.Context.ResponseInfo[KrProcessSharedExtensions.KrProcessClientCommandInfoMark] = commands;
	}

	commands.Add(
		new KrProcessClientCommand(
			DefaultCommandTypes.OpenCard,
			new Dictionary<string, object>(StringComparer.Ordinal)
			{
				[KrConstants.Keys.NewCardID] = dialogCard.ID,
				[KrConstants.Keys.TypeID] = dialogCard.TypeID,
				[KrConstants.Keys.TypeName] = dialogCard.TypeName,
			}).GetStorage());

	// Установка инициализированной карточки служебной записки в качестве основной карточки процесса.
	this.Context.SetMainCard(dialogCard.ID);
}

#region Test support members
#region Constants

/// <summary>
/// Строка в сообщении валидации соответствующая успешному выполнению теста.
/// </summary>
private const string PassedStr = "Passed";

/// <summary>
/// Строка в сообщении валидации соответствующая ошибке при выполнении теста.
/// </summary>
private const string FailedStr = "Failed";

/// <summary>
/// Имя ключа, по которому в <see cref="CardInfoStorageObject.Info"/> карточки, в которой запущен бизнес-процесс, содержится значение флага, показывающего, запущен процесс из тестов или нет. Значение типа: <see cref="bool"/>.
/// </summary>
private const string IsLaunchedInTestKey = "IsLaunchedInTest";

/// <summary>
/// Имя ключа, по которому в <see cref="CardInfoStorageObject.Info"/> карточки, в которой запущен бизнес-процесс, содержится метод инициализации бизнес-процесса при выполнении из тестов. Значение типа: <see cref="Func{T, TResult}"/>, где T - <see cref="WorkflowEngineCompiledBase"/>, TResult - <see cref="ValueTask"/>.
/// </summary>
private const string TestInitializerActionKey = "TestInitializerAction";

#endregion

#region Properties

/// <summary>
/// Возвращает значение, показывающее, что процесс запущен из тестов.
/// </summary>
protected bool IsLaunchedInTest => this.ProcessHash.TryGet<bool>(IsLaunchedInTestKey);

#endregion

#region Protected Methods

/// <summary>
/// Добавляет в результаты валидации сообщение содержащее текст с признаком успешного выполнения теста "<see cref="PassedStr"/>[ <paramref name="suffix"/>]".
/// </summary>
/// <param name="suffix">Строка добавляемая к <see cref="PassedStr"/>.</param>
/// <remarks>Не выполняет действий, если процесс выполняется не из тестов.</remarks>
protected void Passed(string suffix = default)
{
    if (!this.IsLaunchedInTest)
    {
        return;
    }

    var str = PassedStr;

    if (!string.IsNullOrEmpty(suffix))
    {
        str += " " + suffix;
    }

    this.AddInfo(str);
}

/// <summary>
/// Добавляет в результаты валидации сообщение типа <see cref="ValidationResultType.Error"/> с сообщением об успешном выполнении.
/// </summary>
/// <remarks>
/// Созданное сообщение предназначено для остановки выполнения бизнес-процесса. Для этого при проверке результатов выполнения необходимо в методе <see cref="T:Tessa.Test.Default.Shared.Workflow.WeAssert.Passed"/> разрешить наличие ошибок в результате валидации.<para/>
/// Не выполняет действий, если процесс выполняется не из тестов.
/// </remarks>
protected void PassedWithStop()
{
    if (!this.IsLaunchedInTest)
    {
        return;
    }

    this.AddError(PassedStr);
}

/// <summary>
/// Добавляет в результаты валидации сообщение содержащее текст с признаком ошибки в тесте "<see cref="FailedStr"/>[ <paramref name="suffix"/>]".
/// </summary>
/// <param name="suffix">Строка добавляемая к <see cref="FailedStr"/>.</param>
/// <remarks>Не выполняет действий, если процесс выполняется не из тестов.</remarks>
protected void Failed(string suffix = default)
{
    if (!this.IsLaunchedInTest)
    {
        return;
    }

    this.AddError(FailedStr + " " + suffix);
}

/// <summary>
/// Инициализирует бизнес-процесс при выполнении из тестов.
/// </summary>
/// <returns>Асинхронная задача.</returns>
/// <remarks>Не выполняет действий, если процесс выполняется не из тестов.</remarks>
protected async ValueTask InitProcessAsync()
{
    var card = this.StoreCardObject;

    if(card is null)
    {
        return;
    }

    var info = card.TryGetInfo();
    
    if (info != null
        && info.Remove(IsLaunchedInTestKey, out var isLaunchedInTestObj)
        && ((bool) isLaunchedInTestObj))
    {
        this.ProcessHash.Add(IsLaunchedInTestKey, BooleanBoxes.True);

        if (info.Remove(TestInitializerActionKey, out object testInitializerFuncAsyncObj))
        {
            if(testInitializerFuncAsyncObj != null)
            {
                var initFuncAsync = (Func<WorkflowEngineCompiledBase, ValueTask>) testInitializerFuncAsyncObj;
                await initFuncAsync(this);
            }
        }
    }
}

#endregion
#endregion
